bl_info = {
    "name": "SVG23D",
    "blender": (2, 80, 0),
    "category": "Object",
}

import bpy
from math import radians
from bpy_extras.io_utils import ImportHelper
import os
import subprocess
import sys
from bpy.types import Panel

class SVGImportProperties(bpy.types.PropertyGroup):
    svg_filepath: bpy.props.StringProperty(
        name="SVG File",
        subtype='FILE_PATH',
        description="Select an SVG file to import into Blender",
        maxlen=1024,
        default="",
    )

    location_y_multiplier: bpy.props.FloatProperty(
        name="Y Location Multiplier",
        default=1.0,
        min=1,
        description="Adjust the distance of the background plane from the design. Higher values move it further back",
        update=lambda self, context: self.update_plane_location(),
    )

    background_color: bpy.props.FloatVectorProperty(
        name="Background Color",
        default=(1.0, 1.0, 1.0, 1.0),
        subtype='COLOR',
        size=4,
        min=0.0,
        max=1.0,
        description="Choose the color for the background plane",
        update=lambda self, context: self.update_plane_color(),
    )

    extrude_value: bpy.props.FloatProperty(
        name="Extrude",
        default=0.0,
        min=0.0,
        step=0.01,
        description="Add thickness to your design. Higher values make it thicker",
        update=lambda self, context: self.update_extrude_value(),
    )

    empty_name: bpy.props.StringProperty(
        name="Empty Name",
        default="Root",
        description="Technical name for the animation control object",
    )

    glass_color: bpy.props.FloatVectorProperty(
        name="Glass Color",
        default=(1.0, 1.0, 1.0, 1.0),
        subtype='COLOR',
        size=4,
        min=0.0,
        max=1.0,
        description="Choose the main color for your design",
        update=lambda self, context: self.update_glass_color(),
    )

    roughness_value: bpy.props.FloatProperty(
        name="Roughness",
        default=0.2,
        min=0.0,
        max=1.0,
        description="Adjust how smooth or rough the material looks. 0 is perfectly smooth, 1 is very rough",
        update=lambda self, context: self.update_roughness(),
    )

    transparency_value: bpy.props.FloatProperty(
        name="Transparency",
        default=1.0,
        min=0.0,
        max=1.0,
        description="Control how see-through the material is. 0 is completely transparent, 1 is solid",
        update=lambda self, context: self.update_transparency(),
    )

    metallic_value: bpy.props.FloatProperty(
        name="Metallic",
        default=0.0,
        min=0.0,
        max=1.0,
        description="Make the material look more like metal. 0 is non-metallic, 1 is fully metallic",
        update=lambda self, context: self.update_metallic(),
    )

    scale_factor: bpy.props.FloatProperty(
        name="Scale Factor",
        default=1.0,
        min=0.01,
        max=10.0,
        description="Make your design bigger or smaller. Values above 1 make it larger, below 1 make it smaller",
        update=lambda self, context: self.update_scale(),
    )

    is_rendering: bpy.props.BoolProperty(
        name="Is Rendering",
        default=False
    )

    render_progress: bpy.props.FloatProperty(
        name="Render Progress",
        default=0.0,
        min=0.0,
        max=100.0,
        subtype='PERCENTAGE'
    )

    white_plane_exists: bpy.props.BoolProperty(
        name="White Plane Exists",
        default=False
    )

    colored_plane_exists: bpy.props.BoolProperty(
        name="Colored Plane Exists",
        default=False
    )

    def update_scale(self):
        if "Design" in bpy.data.objects:
            bpy.data.objects["Design"].scale = (self.scale_factor, self.scale_factor, self.scale_factor)

    def update_metallic(self):
        if "Design" in bpy.data.objects:
            bpy.data.objects["Design"].data.materials[-1].node_tree.nodes["Principled BSDF"].inputs["Metallic"].default_value = self.metallic_value

    def update_plane_location(self):
        for obj in bpy.data.objects:
            if obj.name.startswith(("WhitePlane", "ColoredPlane")):
                obj.location.y = self.location_y_multiplier

    def update_plane_color(self):
        bgcolor = self.background_color
        if "BackgroundMaterial" in bpy.data.materials:
            material = bpy.data.materials["BackgroundMaterial"]
            if "Principled BSDF" in material.node_tree.nodes:
                material.node_tree.nodes["Principled BSDF"].inputs["Base Color"].default_value = bgcolor

    def update_extrude_value(self):
        if "Design" in bpy.data.objects:
            bpy.data.objects["Design"].data.extrude = self.extrude_value

    def update_glass_color(self):
        if "Design" in bpy.data.objects:
            gcolor = self.glass_color
            bpy.data.objects["Design"].data.materials[-1].node_tree.nodes["Principled BSDF"].inputs["Base Color"].default_value = gcolor

    def update_roughness(self):
        if "Design" in bpy.data.objects:
            bpy.data.objects["Design"].data.materials[-1].node_tree.nodes["Principled BSDF"].inputs["Roughness"].default_value = self.roughness_value

    def update_transparency(self):
        if "Design" in bpy.data.objects:
            bpy.data.objects["Design"].data.materials[-1].node_tree.nodes["Principled BSDF"].inputs[17].default_value = self.transparency_value


class SVGImporterPanel(bpy.types.Panel):
    bl_label = "SVG23D"
    bl_idname = "PT_SVGImporterPanel"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'SVG23D'
    bl_order = 1

    def draw(self, context):
        layout = self.layout
        props = context.scene.svg_import_props

        layout.operator("object.import_svg", text="Import SVG", icon='IMPORT').filepath = props.svg_filepath
        layout.label(text="Design Settings:", icon='MODIFIER')
        layout.prop(props, "extrude_value", text="Thickness")
        layout.prop(props, "scale_factor", text="Size")
        layout.prop(props, "glass_color", text="Color")
        layout.prop(props, "metallic_value", text="Metallic")
        layout.prop(props, "roughness_value", text="Roughness")
        layout.prop(props, "transparency_value", text="Transparency")
        layout.label(text="Background:", icon='IMAGE_PLANE')

        # Toggle buttons for backgrounds
        if props.white_plane_exists:
            layout.operator("object.add_white_plane", text="Remove White Background", icon='X')
        else:
            layout.operator("object.add_white_plane", text="Add White Background", icon='LIGHT')

        if props.colored_plane_exists:
            layout.operator("object.add_colored_plane", text="Remove Colored Background", icon='X')
        else:
            layout.operator("object.add_colored_plane", text="Add Colored Background", icon='MATERIAL')

        layout.prop(props, "background_color", text="Background Color")
        layout.prop(props, "location_y_multiplier", text="Background Distance")
        layout.label(text="Animation & Render:", icon='RENDER_ANIMATION')
        layout.operator("object.animate_svg", text="Create Animation", icon='ANIM')
        layout.operator("object.select_render_path", text="Set Render Location", icon='FILE_FOLDER')

        if props.is_rendering:
            layout.label(text=f"Rendering: {props.render_progress:.1f}%")
            layout.operator("object.cancel_render", text="Cancel Render", icon='CANCEL')
        else:
            layout.operator("object.start_render", text="Render Animation", icon='RENDER_ANIMATION')

        layout.label(text=" ")
        layout.label(text="Created By: Aimen Zaied")

class ImportSVGOperator(bpy.types.Operator, ImportHelper):
    bl_idname = "object.import_svg"
    bl_label = "Import SVG and Transform"
    bl_description = "Choose an SVG file to import into your scene"
    bl_options = {'REGISTER', 'UNDO'}

    filename_ext = ".svg"
    filter_glob: bpy.props.StringProperty(
        default="*.svg",
        options={'HIDDEN'},
        maxlen=255,
    )

    def execute(self, context):
        if not self.filepath:
            self.report({'ERROR'}, "No SVG file selected.")
            return {'CANCELLED'}

        # Store the filepath in the properties
        context.scene.svg_import_props.svg_filepath = self.filepath

        bpy.ops.import_curve.svg(filepath=self.filepath)

        curves = [obj for obj in bpy.context.scene.objects if obj.type == 'CURVE']

        bpy.ops.object.select_all(action='DESELECT')
        for curve in curves:
            curve.select_set(True)

        if curves:
            bpy.context.view_layer.objects.active = curves[-1]

        bpy.ops.object.join()

        # Set origin to geometry
        bpy.ops.object.origin_set(type='ORIGIN_GEOMETRY', center='MEDIAN')

        # Get 3D cursor location
        cursor_loc = bpy.context.scene.cursor.location

        # Move object to cursor location
        bpy.context.active_object.location = cursor_loc

        # Delete existing materials
        bpy.ops.object.material_slot_remove()

        # Delete existing cameras
        existing_cameras = [obj for obj in bpy.data.objects if obj.type == 'CAMERA']
        for camera in existing_cameras:
            bpy.data.objects.remove(camera, do_unlink=True)

        # Assign a new material
        material = bpy.data.materials.new(name="NewMaterial")
        bpy.context.object.data.materials.append(material)
        material.use_nodes = True

        # Set object rotation
        bpy.context.object.rotation_euler = (radians(90), 0, 0)

        # Apply scaling from UI
        scale_factor = context.scene.svg_import_props.scale_factor
        bpy.context.object.scale = (scale_factor, scale_factor, scale_factor)

        bpy.context.object.name = "Design"

        return {'FINISHED'}

class AddWhitePlane(bpy.types.Operator):
    bl_idname = "object.add_white_plane"
    bl_label = "Toggle White Background"
    bl_description = "Add or remove white emissive background plane"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.svg_import_props

        if props.white_plane_exists:
            # Remove the white plane
            for obj in bpy.data.objects:
                if obj.name.startswith("WhitePlane"):
                    bpy.data.objects.remove(obj, do_unlink=True)
            props.white_plane_exists = False
            return {'FINISHED'}

        # Create plane
        bpy.ops.mesh.primitive_plane_add(size=2, enter_editmode=False, align='WORLD', location=(0, props.location_y_multiplier, 0), scale=(1, 1, 1))
        plane = bpy.context.active_object
        plane.name = "WhitePlane"

        # Rotate plane
        bpy.ops.transform.rotate(value=1.5708, orient_axis='X')

        # Create new material
        material = bpy.data.materials.new(name="WhiteEmissiveMaterial")
        plane.data.materials.append(material)
        material.use_nodes = True

        # Clear default nodes
        material.node_tree.nodes.clear()

        # Create emission node
        emission = material.node_tree.nodes.new('ShaderNodeEmission')
        emission.inputs["Color"].default_value = (1, 1, 1, 1)
        emission.inputs["Strength"].default_value = 100

        # Create output node
        material_output = material.node_tree.nodes.new('ShaderNodeOutputMaterial')

        # Link nodes
        material.node_tree.links.new(emission.outputs[0], material_output.inputs[0])

        # Set visibility
        plane.visible_camera = True
        plane.visible_diffuse = False
        plane.visible_glossy = False
        plane.visible_transmission = False
        plane.visible_volume_scatter = False
        plane.visible_shadow = False

        props.white_plane_exists = True

        return {'FINISHED'}

class AddColoredPlane(bpy.types.Operator):
    bl_idname = "object.add_colored_plane"
    bl_label = "Toggle Colored Background"
    bl_description = "Add or remove colored background plane"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        props = context.scene.svg_import_props

        if props.colored_plane_exists:
            # Remove the colored plane
            for obj in bpy.data.objects:
                if obj.name.startswith("ColoredPlane"):
                    bpy.data.objects.remove(obj, do_unlink=True)
            props.colored_plane_exists = False
            return {'FINISHED'}

        # Create plane
        bpy.ops.mesh.primitive_plane_add(size=2, enter_editmode=False, align='WORLD', location=(0, props.location_y_multiplier, 0), scale=(1, 1, 1))
        plane = bpy.context.active_object
        plane.name = "ColoredPlane"

        # Rotate plane
        bpy.ops.transform.rotate(value=1.5708, orient_axis='X')

        # Create new material
        material = bpy.data.materials.new(name="BackgroundMaterial")
        plane.data.materials.append(material)
        material.use_nodes = True

        # Get background color from properties
        bgcolor = context.scene.svg_import_props.background_color

        # Set color in Principled BSDF
        material.node_tree.nodes["Principled BSDF"].inputs["Base Color"].default_value = bgcolor

        props.colored_plane_exists = True

        return {'FINISHED'}

class AnimateSVGOperator(bpy.types.Operator):
    bl_idname = "object.animate_svg"
    bl_label = "Create Animation"
    bl_description = "Set up a rotating animation of your design (2 rotations over 10 seconds)"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        # Set render settings
        bpy.context.scene.render.fps = 30
        bpy.context.scene.render.resolution_x = 1080
        bpy.context.scene.render.resolution_y = 1080
        bpy.context.scene.cycles.samples = 200

        # Set output format to FFmpeg video
        bpy.context.scene.render.image_settings.file_format = 'FFMPEG'
        bpy.context.scene.render.ffmpeg.format = 'MPEG4'
        bpy.context.scene.render.ffmpeg.codec = 'H264'

        # Create an empty object named "Root" if it doesn't exist
        empty_name = "Root"
        if empty_name not in bpy.data.objects:
            bpy.ops.object.empty_add()
            bpy.context.object.name = empty_name
            bpy.context.object.location = (0, 0, 0)
            bpy.context.object.scale = (0.1, 0.1, 0.1)

        # Set keyframe for Z rotation of the "Root" object on frame 1
        bpy.data.objects[empty_name].rotation_euler = (0, 0, radians(0))
        bpy.data.objects[empty_name].keyframe_insert(data_path="rotation_euler", index=2, frame=1)

        # Set keyframe for Z rotation of the "Root" object on frame 300
        bpy.data.objects[empty_name].rotation_euler = (0, 0, radians(-720))
        bpy.data.objects[empty_name].keyframe_insert(data_path="rotation_euler", index=2, frame=300)

        # Set all keyframes to linear interpolation
        for fcurve in bpy.data.objects[empty_name].animation_data.action.fcurves:
            for keyframe in fcurve.keyframe_points:
                keyframe.interpolation = 'LINEAR'
                keyframe.handle_left_type = 'VECTOR'
                keyframe.handle_right_type = 'VECTOR'

        # Set up parenting
        bpy.ops.object.select_all(action='DESELECT')
        bpy.data.objects["Design"].select_set(True)
        bpy.data.objects[empty_name].select_set(True)
        bpy.context.view_layer.objects.active = bpy.data.objects[empty_name]
        bpy.ops.object.parent_set(type='OBJECT')

        # Additional code for camera creation
        bpy.ops.object.camera_add(enter_editmode=False, align='WORLD', location=(0, -0.75, 0), rotation=(radians(90), 0, 0))
        bpy.data.cameras[-1].lens = 150

        return {'FINISHED'}

class SelectRenderPathOperator(bpy.types.Operator, ImportHelper):
    bl_idname = "object.select_render_path"
    bl_label = "Choose Render Output Location"
    bl_description = "Select where to save your rendered animation"

    filename_ext = ""
    use_filter_folder = True

    def execute(self, context):
        output_path = os.path.dirname(self.filepath)
        if output_path:
            # Ensure the directory exists
            if not os.path.exists(output_path):
                os.makedirs(output_path)

            # Set the render output path
            bpy.context.scene.render.filepath = os.path.join(output_path, "render_")
            self.report({'INFO'}, f"Render output path set to: {output_path}")
        else:
            self.report({'ERROR'}, "Please select an output directory")
            return {'CANCELLED'}

        return {'FINISHED'}

def render_progress_update():
    if bpy.context.scene.svg_import_props.is_rendering:
        current_frame = bpy.context.scene.frame_current
        total_frames = bpy.context.scene.frame_end
        progress = (current_frame / total_frames) * 100
        bpy.context.scene.svg_import_props.render_progress = progress
        return 0.1
    return None

class StartRenderOperator(bpy.types.Operator):
    bl_idname = "object.start_render"
    bl_label = "Start Render"
    bl_description = "Start rendering the animation (this might take a while)"
    bl_options = {'REGISTER', 'UNDO'}

    _timer = None

    def modal(self, context, event):
        if event.type == 'TIMER':
            # Check if rendering is still in progress
            if not context.scene.svg_import_props.is_rendering:
                self.cancel(context)
                return {'FINISHED'}

            # Update progress
            current_frame = context.scene.frame_current
            total_frames = context.scene.frame_end
            context.scene.svg_import_props.render_progress = (current_frame / total_frames) * 100

        return {'PASS_THROUGH'}

    def execute(self, context):
        if not bpy.context.scene.render.filepath:
            self.report({'ERROR'}, "Please set the render output location first")
            return {'CANCELLED'}

        context.scene.svg_import_props.is_rendering = True
        context.scene.svg_import_props.render_progress = 0.0

        # Add the timer
        wm = context.window_manager
        self._timer = wm.event_timer_add(0.1, window=context.window)
        wm.modal_handler_add(self)

        # Start the render in the background
        bpy.ops.render.render('INVOKE_DEFAULT', animation=True)

        return {'RUNNING_MODAL'}

    def cancel(self, context):
        if self._timer is not None:
            context.window_manager.event_timer_remove(self._timer)
        context.scene.svg_import_props.is_rendering = False
        context.scene.svg_import_props.render_progress = 0.0

class CancelRenderOperator(bpy.types.Operator):
    bl_idname = "object.cancel_render"
    bl_label = "Cancel Render"
    bl_description = "Cancel the current render"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        bpy.ops.render.render('INVOKE_DEFAULT', animation=False)
        context.scene.svg_import_props.is_rendering = False
        context.scene.svg_import_props.render_progress = 0.0
        return {'FINISHED'}

def register():
    bpy.utils.register_class(SVGImportProperties)
    bpy.utils.register_class(SVGImporterPanel)
    bpy.utils.register_class(ImportSVGOperator)
    bpy.utils.register_class(AddWhitePlane)
    bpy.utils.register_class(AddColoredPlane)
    bpy.utils.register_class(AnimateSVGOperator)
    bpy.utils.register_class(SelectRenderPathOperator)
    bpy.utils.register_class(StartRenderOperator)
    bpy.utils.register_class(CancelRenderOperator)

    bpy.types.Scene.svg_import_props = bpy.props.PointerProperty(type=SVGImportProperties)

def unregister():
    bpy.utils.unregister_class(SVGImportProperties)
    bpy.utils.unregister_class(SVGImporterPanel)
    bpy.utils.unregister_class(ImportSVGOperator)
    bpy.utils.unregister_class(AddWhitePlane)
    bpy.utils.unregister_class(AddColoredPlane)
    bpy.utils.unregister_class(AnimateSVGOperator)
    bpy.utils.unregister_class(SelectRenderPathOperator)
    bpy.utils.unregister_class(StartRenderOperator)
    bpy.utils.unregister_class(CancelRenderOperator)

    del bpy.types.Scene.svg_import_props

if __name__ == "__main__":
    register()
